﻿using System;
using System.DirectoryServices.AccountManagement;
using System.DirectoryServices.ActiveDirectory;
using System.Security.Principal;
using System.Web;
using System.Web.Mvc;
using HIPS.CommonSchemas;

namespace HIPS.Web.UI.Helpers
{
    /// <summary>
    /// Common extension methods for MVC Controllers
    /// </summary>
    public static class ControllerExtensions
    {
        public static ActionResult FileInlineInferMime(this Controller c, byte[] data, string filename)
        {
            // Return file based on inferred MIME type from filename (rather than for example AcceptTypes)
            // NB: GetMimeMapping is a .NET 4.5 method
            return c.FileInline(data, System.Web.MimeMapping.GetMimeMapping(filename), filename);
        }

        public static ActionResult FileInline(this Controller c, byte[] data, string contentType, string filename)
        {
            // Attempt to display "inline" (i.e. render in browser rather than file download)
            // See: http://stackoverflow.com/questions/5826649/returning-a-file-to-view-download-in-mvc
            var cd = new System.Net.Mime.ContentDisposition
            {
                FileName = filename,
                Inline = true,
            };
            c.Response.AppendHeader("Content-Disposition", cd.ToString());

            return new FileContentResult(data, contentType);
        }

        public static void SetAjaxErrorResponseCode(this Controller c)
        {
            if (c.Response != null)
            {
                // Unprocessable Entity
                c.Response.StatusCode = 422; // Also prevents OutputCache
                c.Response.TrySkipIisCustomErrors = true;
            }
        }

        public static ActionResult SkipCacheAsError(this Controller c, ActionResult a)
        {
            c.SetAjaxErrorResponseCode();
            return a;
        }

        internal static UserDetails GetCurrentUserDetails(this Controller c)
        {
            IIdentity i = null;

            //Attempt to get identity from controller.
            if ((c.User != null) && (c.User.Identity != null))
            {
                i = c.User.Identity;
            }

            //If controller identity not present, fall-back to thread identity.
            if (i == null)
            {
                i = System.Threading.Thread.CurrentPrincipal.Identity;
            }

            if (!i.IsAuthenticated)
            {
                throw new InvalidOperationException("User login is required."); // TODO: Handle this differently?
            }

            var domain = i.Name.Contains("\\") ? i.Name.Split('\\')[0] : "";
            var loginName = i.Name.Replace(domain + "\\", String.Empty);
            string displayName;

            /* Retrieve display name */
            /* Use try/catch to deal with remote AD or other issues */
            try
            {
                using (PrincipalContext pc = new PrincipalContext(ContextType.Domain))
                {
                    UserPrincipal up = UserPrincipal.FindByIdentity(pc, loginName);

                    if (up == null)
                    {
                        throw new InvalidOperationException("Could not find user principal.");
                    }

                    // Display name can be null, fall back to login name
                    displayName = up.DisplayName ?? loginName.Replace('.', ' ');
                }
            }
            catch (Exception)
            {
                // Couldn't retrieve display name, fall back to login name
                displayName = loginName.Replace('.', ' ');
            }

            return new UserDetails()
            {
                Domain = domain,
                Login = loginName,
                Name = displayName,
                Role = UserRole.InteractiveUser
            };
        }

    }
}